﻿#requires -version 3
<#
    .SYNOPSIS 
      Monitor ISDN interfaces, channel usage and files system usage on one or more gateways
    .Description
      enter 1..n IP Adresses as Parameters
      Send e-mail alert if limits are reached
    .PARAMETER Gateways
      String array of gateways to monitor
    .PARAMETER FileSystemThreshold
      Threshold in percent for file system usage.
      Default value = 95
    .PARAMETER Pause
      Time in seconds to pause between checks.
      Default Value = 30
    .PARAMETER TransportServer
      SMTP Server
    .PARAMETER FromAddress
      'From' e-mail address for alerts
    .PARAMETER SMTPPort
      SMTP Port of SMTP server
    .PARAMETER ToAddress
      'To' e-mail address for alerts
    .PARAMETER GatewayFile
      Path to a file where gateways are listed including credentials (usage example 3)
    .PARAMETER Log
      Activate log in separate file
    .PARAMETER ShowErrors
      Show system error (description)
    .EXAMPLE
      .\Monitor-OMGates.ps1 10.6.1.33
    .EXAMPLE
      .\Monitor-OMGates.ps1 -Gateways 192.168.1.56,omgate2.mydomain.local -FileSystemThreshold 90 -Pause 600 -TransportServer smtp.mydomain.lan -toAddress admin@mydomain.lan
    .EXAMPLE
      .\Monitor-OMGates.ps1 -GatewayFile .\gateways_sample.txt

      the text-file 'gateways_sample.txt' is a csv file which looks like this:
      
      Identity, Password, [User]
      10.4.50.162, omc, root
      10.4.50.160, admin123
      10.4.50.60, cmo, root

      the 'User' column is optional, default Value for User is 'root'. If you create your own GatewayFile you must not type brackets around 'User'
#>

param(
[Parameter(Mandatory=$false)]
    [string[]] $Gateways,
[int] $FileSystemThreshold = 95,
[int] $Pause = 30,
[Parameter(Mandatory=$false)]
    [string] $TransportServer = "mailserver.mydomain.local",
[Parameter(Mandatory=$false)]
    [string] $FromAddress = "officemastergate@mydomain.local",
[Parameter(Mandatory=$false)]
    [int] $SMTPPort = 25,
[Parameter(Mandatory=$false)]
    [string] $ToAddress ="admin@mydomain.local",
[Parameter (Mandatory=$false)]
    [string] $GatewayFile,
[Parameter(Mandatory=$false)]
    [switch] $Log,
[Parameter(Mandatory=$false)]
    [switch] $ShowErrors
)

Import-Module OfficeMaster


# after sending e-mail suppress further e-mails for this gateway for nn minutes
$alert_repeat_minutes = 60

# log file name
$LogOutputFileLocation = "Monitor-OMGates.log"


# version definitions
$FileSystemCmdletMinVersion = 200000002
$FileSystemFirmwareMinVersion = 40000000174
$FileSystemCheckSupported = $false

# default definitions
$gatewayDefaults = @{"Password" = "omc"; "User" = "root"; "LastErrorTime" = Get-Date "01.01.2010"}
$gatewayObject = @()
$idx = 0
$indent = "  "



# check for 'q' key to exit operation
#
function Stop-It
{
    if ($Host.UI.RawUI.KeyAvailable -and ("q" -eq $Host.UI.RawUI.ReadKey("IncludeKeyUp,NoEcho").Character)) 
    {
        Write-Host "Exiting now..." -Background DarkRed
        $stop = $true
    }
    else
    {
        $stop = $false
    }
    $stop
}



function Send-Alert($subject, $body, $idx) 
{
    $ts = New-TimeSpan $gatewayObject[$idx].LastErrorTime (Get-Date)

    if ($ts.TotalMinutes -ge $alert_repeat_minutes)
    {
        try
        {
            Send-MailMessage -SmtpServer $TransportServer -body $body -from $FromAddress -Subject $subject -To $ToAddress -Port $SMTPPort -ErrorAction Stop
        }
        catch
        {
            Write-Host -Background DarkRed "could not send alert mail - check settings "
            if ($ShowErrors)
            {
                Write-Host -Background DarkRed $Error[0]
            }
            return
        }
        $gatewayObject[$idx].LastErrorTime = Get-Date
    }
}


function Start-Log
{
    try
    {
        Stop-Transcript -ErrorAction SilentlyContinue | Out-Null
    }
    catch
    {
        
    }
    $MyInvocation.PSScriptRoot
    Start-Transcript -path $LogOutputFileLocation -append
}


function Stop-Log
{
    #if users exits script with crtl+c this function will not be called
    Stop-Transcript | out-null
}


function Prepare-GatewayObject
{
    if ($Gateways)
    {
        #array of gateways was passed
        Write-Debug "Add password and user to gateway object"

        foreach($gateway in $Gateways)
        {
            $gatewaySingleObject = New-Object System.Object
            $gatewaySingleObject | Add-Member -Type NoteProperty -Name Identity -Value $gateway
            $gatewaySingleObject | Add-Member -Type NoteProperty -Name User -Value $gatewayDefaults.User
            $gatewaySingleObject | Add-Member -Type NoteProperty -Name Password -Value $gatewayDefaults.Password
            $gatewaySingleObject | Add-Member -Type NoteProperty -Name LastErrorTime -Value $gatewayDefaults.LastErrorTime
            $Script:gatewayObject += $gatewaySingleObject
            
        }
    }
    elseif ($GatewayFile.Length -gt 0) {
        
        #file of gateways was passed

        Write-Debug "Import gateways, users and passwords from file to gateway object"

        $gatewayArray = Import-Csv $GatewayFile

        foreach ($gateway in $gatewayArray)
        {
            $gateway | Add-Member -Type NoteProperty -Name LastErrorTime -Value $gatewayDefaults.LastErrorTime
            
            if ($gateway.Password.Length -eq 0)
            {
                #set standard password
                $gateway.Password = $gatewayDefaults.Password
            }
            if ($gateway.User.Length -eq 0)
            {
                #set standard user
                $gateway.User = $gatewayDefaults.User
            }

            $Script:gatewayObject += $gateway
        }
    }
    else {
        
        #nothing was passed, prompt for input
        $ip = ""
        $counter = 0
        Write-Host "Gateways to monitor:"
        do
        {
            $ip = Read-Host "gateway[$counter]: "
            if ($ip.Length -gt 0)
            {
                $gatewaySingleObject = New-Object System.Object
                $gatewaySingleObject | Add-Member -Type NoteProperty -Name Identity -Value $ip
                $gatewaySingleObject | Add-Member -Type NoteProperty -Name User -Value $gatewayDefaults.User
                $gatewaySingleObject | Add-Member -Type NoteProperty -Name Password -Value $gatewayDefaults.Password
                $gatewaySingleObject | Add-Member -Type NoteProperty -Name LastErrorTime -Value $gatewayDefaults.LastErrorTime
                $Script:gatewayObject += $gatewaySingleObject
            }
            $counter += 1
        }
        while($ip.Length -gt 0)
    }
}


function Check-Versions
{
    Write-Host "Check versions of cmdlets and gateways..."
    
    #check version of omg filesystem and omg cmdlet
    $cmdVersion = Get-OmgCmdletVersion
    $cmdVersion = $cmdVersion.Split('.')

    $majorversion = "{0:D4}" -f [int]$cmdVersion[0]
    $minorVersion = "{0:D4}" -f [int]$cmdVersion[1]
    $revisonVersion = "{0:D4}" -f [int]$cmdVersion[2]

    $cmdVersion = [int64] ($majorversion + $minorVersion + $revisonVersion)
    if ($cmdVersion -lt $FileSystemCmdletMinVersion)
    {
        Write-Host "Not all features are supported. Please install the newest version of the OfficeMaster CMDlets!" -ForegroundColor Red -BackgroundColor Yellow
        $Script:FileSystemCheckSupported = $false
    }
    else
    {
        $Script:FileSystemCheckSupported = $true
    }

    #check omg version
    foreach ($gate in $gatewayObject)
    {
        #get version
        try
        {
            $gateVersion = Get-OmVersion -Identity $gate.Identity -UserName $gate.User -UserPassword $gate.Password
        
            if ($gateVersion.Value[$gateVersion.Name.IndexOf("UserFS")].Length -eq 0)
            {
               Write-Host -Background DarkRed "Could not reach '" $gate.Identity "' to get firmware version - check credentials!"
               $Subject = "Gateway $id unreachable"
               $Body = "Gateway $id is unreachable for firmware version check - check the credentials"
               Send-Alert $Subject $Body $idx
               continue 
            }
         }
         catch
         {
            Write-Host -Background DarkRed "Could not reach '" $gate.Identity "' to get firmware version - check credentials!"
            $Subject = "Gateway $id unreachable"
            $Body = "Gateway $id is unreachable for firmware version check - check the credentials"
            Send-Alert $Subject $Body $idx
            continue
         }

        $userFSIndex = $gateVersion.Name.IndexOf("UserFS")

        $version = $gateVersion.Value[$userFSIndex]

        $majorVersion = $version.Substring(0,$version.IndexOf('.'))
        $version = $version.Substring($majorVersion.Length+1)
        $minorVersion = $version.Substring(0,$version.IndexOf('-'))
        $version = $version.Substring($minorVersion.Length+1)
        if ($version.IndexOf('.') -ne -1)
        {
            $buildNumber = $version.Substring(0,$version.IndexOf('.'))
        }
        else
        {
            $buildNumber= $version
        }

        $majorVersion = "{0:D4}" -f [int]$majorVersion
        $minorVersion = "{0:D4}" -f [int]$minorVersion
        $buildNumber = "{0:D6}" -f [int]$buildNumber

        $version = [int64] ($majorVersion + $minorVersion + $buildNumber)

        if ($version -lt $FileSystemFirmwareMinVersion)
        {
            $gate | Add-Member -Type NoteProperty -Name FsSupported -Value $false

            Write-Host "Not all features are supported. Please consider to update the firmware on gateway '" $gate.Identity "'!" -BackgroundColor DarkRed
        }
        else
        {
            $gate | Add-Member -Type NoteProperty -Name FsSupported -Value $true
        }
    }
}


function Check-Filesystem($idx)
{
    $id = $gatewayObject[$idx].Identity
    $user = $gatewayObject[$idx].User
    $pw = $gatewayObject[$idx].Password

    Write-Host -NoNewline "$indent file system usage: "

    if ($Script:FileSystemCheckSupported -eq $true -and $gatewayObject[$idx].FsSupported -eq $true)
    {
        try
        {
            $fs = Get-OmFileSystemInfo -Identity $id -UserName $user -UserPassword $pw
        }
        catch
        {
            Write-Host -Background DarkRed "unreachable"
            $Subject = "Gateway $id unreachable"
            $Body = "Gateway $id is unreachable for file system check"
            Send-Alert $Subject $Body $idx
            return
        }


        foreach($i in $fs) 
        {
            $percent_str = $i.PercentUsed.TrimEnd("%")
            if ($percent_str.Count -ne 0) 
            {
                if ($i.MountPoint -eq "/tmp"-or $i.MountPoint -eq "/data") 
                {
                    [int]$percent = [convert]::ToInt32($percent_str, 10)

                    if ($percent -ge $FileSystemThreshold) 
                    {
                        Write-Host -NoNewline -Background DarkRed $i.MountPoint"`b:"$percent"% "
                        $Subject = "Filesystem limit reached at Gateway $id"
                        $Body = "File system: $($i.MountPoint), Usage: $($i.PercentUsed)"
                        Send-Alert $Subject $Body $idx
                    }
                    else
                    {
                        Write-Host $i.MountPoint"`b:"$percent"% " -ForegroundColor Green -NoNewline

                    }
                    
                }
            }
        }
    }
    else
    {
        Write-Host -BackgroundColor DarkRed "not supported" -NoNewline
    }
    Write-Host ""
}



function Check-TotalCalls($idx) 
{
    $prichannelsavailable = 0
    $prichannelsused = 0;
    $pcmchannelsavailable = 0
    $pcmchannelsused = 0;
    $brichannelsavailable = 0
    $brichannelsused = 0
    $id = $gatewayObject[$idx].Identity
    $user = $gatewayObject[$idx].User
    $pw = $gatewayObject[$idx].Password

    Write-Host -NoNewline "$indent channel usage....:"

    try
    {
	    $omgate = Get-OmInterfaceCallState -Identity $id -UserName $user -UserPassword $pw
    }
    catch
    {
        Write-Host -Background DarkRed "unreachable"
        $Subject = "Gateway $id unreachable"
        $Body = "Gateway $id is unreachable for total calls check"
        Send-Alert $Subject $Body $idx
        return
    }

	foreach ($element in $omgate) 
    {
        if ($element.Interface.Endswith("PRI") -and $element.State -ne "disabled")
        {
            $prichannelsavailable += 30
            $prichannelsused += $element.CallCount
        }
        elseif ($element.Interface.Endswith("PCM") -and $element.State -ne "disabled")
        {
            $pcmchannelsavailable += 30
            $pcmchannelsused += $element.CallCount
        }
        elseif ($element.Interface.EndsWith("BRI") -and $element.State -ne "disabled")
        {
            $brichannelsavailable += 2
            $brichannelsused += $element.CallCount
        }
	}

    if ($prichannelsused -eq $prichannelsavailable -and $prichannelsavailable -ne 0)
    {
        Write-Host -NoNewline -Background DarkRed " PRI" $prichannelsused/$prichannelsavailable
        $Subject = "All channels of gateway $id are busy"
        $Body = "All PRI interfaces reached usage limit"
        Send-Alert $Subject $Body $idx
    }
    elseif ($prichannelsavailable -ne 0)
    {
        Write-Host -NoNewline " PRI" $prichannelsused/$prichannelsavailable -ForegroundColor Green
    }

    if ($pcmchannelsused -eq $pcmchannelsavailable -and $pcmchannelsavailable -ne 0)
    {
        Write-Host " " -NoNewline
        Write-Host -Background DarkRed " PCM" $pcmchannelsused/$pcmchannelsavailable 
        $Subject = "All channels of gateway $id are busy"
        $Body = "All PCM interfaces reached usage limit"
        Send-Alert $Subject $Body $idx
    }
    elseif ($pcmchannelsavailable -ne 0)
    {
        Write-Host -NoNewline " PCM" $pcmchannelsused/$pcmchannelsavailable -ForegroundColor Green
    }

    if ($brichannelsused -eq $brichannelsavailable -and $brichannelsavailable -ne 0)
    {
        Write-Host " " -NoNewline
        Write-Host -Background DarkRed " BRI" $brichannelsused/$brichannelsavailable 
    }
    elseif ($brichannelsavailable -ne 0)
    {
        Write-Host -NoNewline " BRI" $brichannelsused/$brichannelsavailable -ForegroundColor Green
    }

    Write-Host ""
}



function Check-Interfaces($idx)
{
    $all_ok = $true
    $id = $gatewayObject[$idx].Identity
    $user = $gatewayObject[$idx].User
    $pw = $gatewayObject[$idx].Password

    Write-Host -NoNewline "$indent interface status.: "

    try
    {
        $omgate = Get-OmInterfaceState -Identity $id -UserName $user -UserPassword $pw
    }
    catch
    {
        Write-Host -Background DarkRed "unreachable"
        $Subject = "Gateway $id unreachable"
        $Body = "Gateway $id is unreachable for interfaces check"
        Send-Alert $Subject $Body $idx
        return
    }

	foreach ($element in $omgate) 
    {
		if ($element.State.EndsWith("L2DN"))
        {
            Write-Host -NoNewline -Background DarkRed "Interface $($element.Interface) is inactive, state=" $element.State
            $Subject = "Layer 2 down at Gateway $id "
            $Body = "Interface $($element.Interface) is inactive"
            Send-Alert $Subject $Body $idx
            $all_ok = $false
        }
	}
    if ($all_ok -eq $true)
    {
        Write-Host "ok" -ForegroundColor Green
    }

}




#############
# main loop #
#############

$enough = $false

if ($Log) { Start-Log }

Write-Host "Press 'q' to stop" -ForegroundColor Yellow

Prepare-GatewayObject

Check-Versions

Write-Debug "'gatewayObject': $gatewayObject"

while ($enough -eq $false)
{
	for ($idx = 0; $idx -lt $gatewayObject.count; $idx++)
	{
    	$date = get-date -UFormat "%d.%m.%Y %H:%M:%S"
        Write-Host "Last Run: " $date " for Gateway " -NoNewline
        Write-Host $gatewayObject[$idx].Identity -ForegroundColor Yellow

        Check-TotalCalls($idx)

        $enough = Stop-It
        if ($enough -eq $true) 
        {
            Write-Host ""
            break
        }

        Check-Filesystem($idx)

        $enough = Stop-It
        if ($enough -eq $true) 
        {
            Write-Host ""
            break
        }

        Check-Interfaces($idx)
        Write-Host ""

        if ($enough -eq $true) 
        {
            break
        }
	}

    if ($enough -ne $true) {

        for ($i = 0; $i -lt $Pause; $i++) 
        { 
            [int]$TimeLeft = $Pause - $i 
            Write-Progress -Activity "Waiting $Pause seconds, press 'q' to stop..." -PercentComplete (100/$Pause * $i) -CurrentOperation "$TimeLeft seconds left ($i elapsed)" -Status "Please wait" 
            Start-Sleep -s 1 
            $enough = Stop-It
            if ($enough -eq $true) 
            {
                break
            }
        } 

        Write-Progress -Completed $true -Status "Please wait" 
    }
}

if ($Log) { Stop-Log }
